unit HmRs232;
{A fake Pascal (Delphi) program module containing what is
 necessary or helpful to control Homer via RS232 interface.
 19-Dec-03  First version
 12-Mar-04  ReceiveDataObject
 18-Sep-06  DecodeMDO
 10-Aug-09  Added PrflIncluded, modified DecodeMDO
 29-Jan-10  Added DecodeMotorPositions; DecodeMDO modification
 12-Mar-19  Added exxlananion of function Dexp(x)}

interface

const
 {Constants}
 {Value of communication error Err00 byte set by ReceiveByte function
  if RS232 timeout occured}
  RXtimeout=6;

 {Command label (128): precedes byte that should be interpreted as
  a command/message code not data (number). If a data byte 128 should be
  sent then the  byte is sent twice(128, 128). Examples:
  1. if you receive ... 128, 28, ... then 28 is the command byte msDataBegin
     saying that a stream of data bytes will follow until next command byte
     occurs. We are interested here only in data streams ended by 128, 16,
     saying that this was a Measurement Data Object (MDO)
  2. if you receive ... 128, 128, ... then it means a single data byte
     with value=128}
  CmndLbl=128;

 {Some command codes}
  msMeasObject=16;
  msDataBegin=28;

  MaxNbuf=16384;  {Max size of RX buffer, here quite exaggerated}
  NdoMax=31;     {Max size of MDO}

type
  TBuf=array[0..MaxNbuf] of byte;   {RX buffer structure}
  TMdo=array[0..NdoMax-1] of byte; {structure for MDO: array of NdoMax bytes}
  complex=record re,im:Tfloat end;
  
  THresults=record
    HomerIncluded,MotorsIncluded:boolean;
    HomerValid:boolean;          {Homer measurement results are valid}
    PulseSerialIncluded:boolean; {MDO is a sample of Rectified/Pulsed waveform}
    PrflIncluded:boolean;        {MDO includes mean reflected  power}
    Fre:longint;                 {Frequency in units of 10 Hz}
    Rho_m:complex;               {Real+imag part of measured refl. coef.}
    Mag_m,Pha_m:double;          {Mag+phase of measured refl. coef.}
    Rho_d:complex;               {Deembedded reflection coefficient}
    Mag_d,Pha_d:double;
    Pinc,Prfl,Pabs:double;       {Incident, reflected and absorbed powers}
    Temp:double;                 {Temperature}
    PulsErr:boolean;             {Pulse measurement has failed}
    Oflow:boolean;               {Overflow of ADCs}
    TooCold:boolean;             {Temperature limits exceeded}
    TooHot:boolean;
    LowSignal:boolean;           {Warning: low signal}
    FreSubst:boolean;            {Warning: counter data invalid}
    FrePast:boolean;             {Previous frequency sent}
    HmInvalid:boolean;           {Homer data invalid}
    PulseSerial:integer;
  end;

var
 {Variables}
  Mdo=TMdo;             {Measurement Data Object - MDO}
  Err00:integer;        {Communication error}
  ErrByte:byte;         {Sent by Homer as Mdo[1] if Homer data included in MDO}
  Mdo_size:word;        {Byte count of the currently received MDO}
  HomResults:THresults; {Results obtained by decoding MDO}

implementation

var
  Finito:boolean; {flag used to stop receiving bytes}

function ReceiveByte:byte;
{Example of Windows receive byte routine. ReadFile and ClearCommError
 functions are Delphi implementations of Win API functions.
 Function waits for a byte from serial link. If nothing comes until
 a set Timeout, it sets Err00 error byte and returns a predefined
 substitute byte. Timeout is set by implementation of SetCommTimeouts
 Win API function. Other Win API functions were used to prepare COM port
 (like CreateFile, GetCommState, SetCommState)}
var
  ok:boolean;
  n:longint;
  rx_byte:byte;
begin
  Err00:=0;
  {Wait for 1 byte, n means how many have actually been read}
  ok:=ReadFile(Hnd,rx_byte,1,n,nil) and (n>0);
  if ok then Result:=rx_byte
  else begin {error occured}
    Err00:=RXtimeout;
    ok:=ClearCommError(Hnd,ComErr,ComStat); {Win API}
    Result:=SubstituteByte;
  end;
end;

procedure ReceiveAndDecodeByte(var b:byte; var IsCommand:boolean);
{Procedure receives byte (B1). If the B1 is equal to command label
 B1=CmndLbl=128 then another byte (B2) is received. If also B2=128
 then it is returned as data byte 128 and IsCommand flag is
 set to false. If B2 differs from B1, B2 is returned and interpreted
 as command code not data. To indicate that, IsCommand flag
 is set to true}
begin
  IsCommand:=false;
  b:=ReceiveByte;  {read one byte from RS232. ReceiveByte routine
    also sets error byte Err00}
  if Err00=0 then if b=CmndLbl then begin {read again}
    b:=ReceiveByte;
    IsCommand:=b<>CmndLbl;
  end
end;

function ReceiveDataObject(var RxBuff; var n:integer; var EndMsg:byte
  ; wait_ms:integer):boolean;
{Waits max wait_ms milliseconds for arrival of msDataBegin command.
  If it comes, resets timeout and receives data until next command;
  this is interpreted as EndMsg}
var
  Buffer:TBuf absolute RxBuff;  {a way of converting structure types}
  WasCommand,DataBeginDetected,Finish:boolean;
  b:byte;
  StartTime,Lasts:longint;
begin
  n:=0; {initialize data buffer counter}
  EndMsg:=255; {for case of failure return a nonexistent EndMsg}
  DataBeginDetected:=false; {set to true when msDataBegin command comes}
  Finish:=false; {reset flag to quit}
  StartTime:=GetTickCount; {initialize timeout timer}
  repeat
    Result:=false;
    ReceiveAndDecodeByte(b,WasCommand);
    if Err00=0 then begin {no comm error occured}
      if WasCommand then begin {byte b is a command not data}
        case b of
          msDataBegin: {the command is data begin: hence consequent received
            bytes will be data. Receive them until the next byte comes that is
            not data byte but command byte. This is end command code EndMsg}
            begin
              n:=0; {initialize data buffer counter}
              DataBeginDetected:=true; {flag allowing to store incoming data}
              StartTime:=GetTickCount; {reset timeout}
            end;
          else if DataBeginDetected then begin {first command byte after
           msDataBegin is end command EndMsg; do finish with success}
            Finish:=true;
            EndMsg:=b;
            Result:=true;
          end;
        end
      end  {of WasCommand}
      else {byte b is data}
      if DataBeginDetected then begin
        {Store data provided msDataBegin came before}
        Buffer[n]:=b;
        inc(n)
      end; 
    end {of case Err00=0}
    else Finish:=true; {communication error}
    Application.ProcessMessages;   {see MainLoopExample}
    Lasts:=GetTickCount-StartTime; {how long it takes}
  until Finish or (Lasts>wait_ms); {end if success or error or timeout}
end; {ReceiveDataObject}

function ReceiveMeasObject:boolean;
{Function receives bytes until command byte msDataBegin comes. Then it
 continues to receive data bytes and stores them to Mdo array until
 an end command byte comes. If the end command byte is msMeasObject then
 data were indeed MDO and all is ok; the function returns Result=true.
 If a communication error occured or the received data packet was not MDO,
 Result=false}
var
  b:byte;
  WasCommand,Finish,DataBeginDetected:boolean;
  n:word;
begin
  b:=0;
  n:=0; {initialize counter of received bytes}
  Finish:=false; {reset flag to quit}
  DataBeginDetected:=false;
  {Repeat receiving and interpreting bytes}
  repeat
    {ReceiveAndDecodeByte routine receives a byte and decides whether it was
     a command byte or data byte. Its internal routines also set communication
     error Err00, which can be among others timeout error. A timeout decision
     could be in principle implemented also in this cycle}
    ReceiveAndDecodeByte(b,WasCommand);
    if Err00=0 then begin  {no comm error occured}
      if WasCommand then begin {byte b is a command not data}
        case b of {what command is it?}
          msDataBegin: {the command is data begin: hence following received
            bytes will be data - but they may or may not be MDO data. Receive
            them until a next byte comes that is not data but command again.
            This end command byte will say whether the data were MDO. Anyway,
            store first NdoMax of the received bytes to Mdo}
            begin
              n:=0; {initialize counter of received bytes}
              DataBeginDetected:=true  {flag allowing to store incoming data}
            end;
          msMeasObject: begin  {this is a terminating command byte for a MDO.
            But we have received complete data packet only if msDataBegin
            command has arrived before, which case is indicated by
            DataBeginDetected=true. In this case we can finish with
            success. The other case means we have missed data begin
            (we started receiving bytes in the middle of relation);
            we go out too (set Finish to true) but with no success.
            Note 1:
            Maybe a better way to deal with the latter (=no succes) case is
            let Finish stay false, then you just continue receiving bytes
            until a new msDataBegin command arrives. We have not tested
            this option}
            Result:=DataBeginDetected;
            Finish:=true; {or, in accordance with Note 1, you may wish
              to modify this command to Finish:=DataBeginDetected}
          end;
        end {case}
      end
      else {b is data byte. If msDataBegin command came before, store it}
        if DataBeginDetected then begin
          if n<NdoMax then begin {store up to NdoMax data bytes}
            Mdo[n]:=b; {store}
            inc(n)     {increase counter}
          end
          else begin {if more than NdoMax data came, this is certainly
            not MDO, finish with no success}
            Result:=false;
            Finish:=true; {or you can leave it false to continue receiving
              and try to catch next msDataBegin command byte}
          end;
        end
    end
    else begin {comm error occured}
      Finish:=true    {quit}
      Result:=false;  {with no success result}
    end;
  until Finish;
  {Fill Status Byte SB: (the command may also be elsewhere, not exactly here)}
  if Result then StatusByte:=Mdo[0];
end; {HM_ReceiveMeasObjec}

function CheckSumCorrect(var Mo:TMdo; n:integer):boolean;
{Verify checksum of object Mo with size n; return true if ok}
var
  i,Csum:integer;
begin
  Csum:=0;
  {Valid Mo bytes are Mo[0]..Mo[n-1], the last one contains
   the sent checksum byte: we must not use it for our checksum
   computations}
  for i:=0 to n-2 do Csum:=Csum+Mo[i]);
  Csum:=Csum and 255;   {Take only LSB}
  Result:=Csum=Mo[n-1]; {Compare}
end;

procedure AnalyzeErrorByte;
{Evaluates errors. Meaning see in THresults type definition}
begin
  with HomResults do begin
    PulsErr  :=ErrByte and 1  =  1;   {Pulse measurement has failed}
    Oflow    :=ErrByte and 2  =  2;   {Overflow of ADCs}
    TooCold  :=ErrByte and 4  =  4;   {Temperature limits exceeded}
    TooHot   :=ErrByte and 8  =  8;
    LowSignal:=ErrByte and 16 = 16;   {Warning: low signal}
    FreSubst :=ErrByte and 32 = 32;   {Warning: counter data invalid}
    HmInvalid:=ErrByte and 64 = 64;   {Homer data for some reason invalid}
  end;
end;

function DecodeMDO:boolean;
{Modified 10-Aug-09, 29-Jan-10}

var
  i:integer;
  Mdo_index:word;

  {Typecasting procedures in Delphi notation}
  function GetSmallPtr(p:pointer):smallint; {signed 16-bit}
  type PSmall=^smallint;
  begin GetSmallPtr:=PSmall(p)^ end;

  function GetLongPtr(p:pointer):longint;  {signed 16-bit}
  type PLong=^longint;
  begin GetLongPtr:=PLong(p)^ end;

  function GetWordPtr(p:pointer):word; {unsigned 16-bit}
  type PWord=^word;
  begin GetWordPtr:=PWord(p)^ end;

  procedure DecodeMeasurementData;
  {Modified 10-Aug-09}
  begin
    with HomResults do begin
      {Meaning see in THresults type definition}
      Pinc:=(256*word(Mdo[2])+Mdo[3])*Dexp(Mdo[4]-10);
      {Function Dexp(x) = 10 to power x, e.g. Dexp(3) = 1000}
      Temp:=GetSmallPtr(@Mdo[5])/10.0;
      Rho_m.re:=GetSmallPtr(@Mdo[8])/4096.0;
      Rho_m.im:=GetSmallPtr(@Mdo[10])/4096.0;
      Fre:=GetLongPtr(@Mdo[12]);
      Rho_d.re:=GetSmallPtr(@Mdo[16])/4096.0;
      Rho_d.im:=GetSmallPtr(@Mdo[18])/4096.0;
      inc(Mdo_index,19);
      if PulseSerialIncluded then begin
        PulseSerial:=GetWordPtr(@Mdo[Mdo_index]);
        inc(Mdo_index,2); {Mdo_index will be 22}
      end
      else if PrflIncluded then begin
        Prfl:=(256*word(Mdo[21])+Mdo[20])*Dexp(Mdo[7]-10);
        inc(Mdo_index,2); {Mdo_index will be 22}
      end;
    end;
  end;

  procedure DecodeMotorPositions;
  {Created 10-Aug-09. Uses MDO and current value of Mdo_index}
  begin
    Mot1_steps:=GetSmallPtr(@Mdo[Mdo_index]);
    Mot2_steps:=GetSmallPtr(@Mdo[Mdo_index+2]);
    Mot3_steps:=GetSmallPtr(@Mdo[Mdo_index+4]);
    MotStatByte1:=Mdo[Mdo_index+6];
    MotStatByte2:=Mdo[Mdo_index+7];
  end;

  procedure DerivedResults;
  begin
    with HomResults do begin
      {Meaning see in THresults type definition}
      Mag_m:=Hypot(Rho_m.re,Rho_m.im);
      Pha_m:=ArcTan2(Rho_m.im,Rho_m.re)*180.0/Pi; {Never use atan(im/re)!!!}
      Mag_d:=Hypot(Rho_d.re,Rho_d.im);
      Pha_d:=ArcTan2(Rho_d.im,Rho_d.re)*180.0/Pi;
      if not PrflIncluded then Prfl:=Pinc*sqr(Mag_m);
      if Prfl>Pinc then Prfl:=Pinc;  {Clipping - passsive loads assumed}
      Pabs:=Pinc-Prfl;
    end;
  end;

begin {DecodeMDO}
  {First thing: determine number of bytes Mdo_size to be able to verify
   the checksum, which is stored in the last byte}
  with HomResults do begin
    {At least one byte is always transmitted = status Mdo[0]}
    Mdo_size:=1;
    {Homer measurement data are present if bit 2 of Mdo[0] is 1}
    HomerIncluded:=Mdo[0] shr 2 and 1 = 1;
    if HomerIncluded then {another 19 bytes are present} begin
      inc(Mdo_size,19);
      ErrByte:=Mdo[1];
      AnalyzeErrorByte(HomResults);
    end;
    PulseSerialIncluded:=HomerIncluded and ((Mdo[0] and 3)<>0);
    PrflIncluded:=(not PulseSerialIncluded)
      and (HomerIncluded and ((Mdo[0] and 64)>0));
    if PrflIncluded or PulseSerialIncluded then inc(Mdo_size,2);
    {Motors data are present if bit 4 of Mdo[0] is 1}
    MotorsIncluded:=Mdo[0] shr 4 and 1 = 1;
    if MotorsIncluded then inc(Mdo_size,8); {another 8 bytes are present}
    HomerValid:=HomerIncluded and not HmInvalid;
    inc(Mdo_size); {last byte is checksum}
    Result:=CheckSumCorrect(Mdo,Mdo_size) and (HomerValid or MotorsIncluded);
    {Previous frequency is sent, i.e. no newly measured value exists}
    FrePast:=Mdo[0] shr 3 and 1 = 1;
    if not Result then exit; {...to avoid various exceptions like division by zero}
    {Now that various flags have been determined and data have been checked
     for integrity (checksum) and correctness (not HmInvalid),
     do the decoding itself}
    Mdo_index:=1;  {temporary pointer to MDO bytes}
    if HomerIncluded then begin
      if HomerValid then begin
        DecodeMeasurementData;
        DerivedResults;
      end
      else begin {do not decode; leave previous results}
        if PulseSerialIncluded or PrflIncluded
          then inc(Mdo_index,21)
          else inc(Mdo_index,19)
      end;
    end;
    if MotorsIncluded then DecodeMotorPositions; {uses current value of Mdo_index}
  end; {with HomResults}
end;

procedure MainLoopExample;
{Just an indication of how it could be}
var
  ok,HomerIncluded,MotorsIncluded:boolean;
begin
  {Clear RS232 Windows buffers - Delphi implem. of Windows API function}
  PurgeComm(Hnd,PURGE_TXABORT+PURGE_TXCLEAR+PURGE_RXABORT+PURGE_RXCLEAR);
  Finito:=false; {reset flag}
  repeat {main loop}
    ok:=ReceiveMeasObject;  {receive MDO}
    {Function DecodeMDO takes Mdo as input and converts its bytes to
     desired quantities. The obtained results are returned in the structure
     HomResults. The DecodeMDO function in this example also returns flags
     HomerIncluded and MotorsIncluded saying wheter valid Homer data
     and motors data were present in MDO, respectively}
    if ok then ok:=DecodeMDO;
    {Now process the obtained data the way you wish to}
    if ok then begin
      if HomerIncluded then DoWhatYouWishWithHomerData;
      if MotorsIncluded then DoWhatYouWishWithMotorsData;
    end;
    {Follows Delphi function giving Windows a chance to respond
     to messages that might have come and are waiting to be attended.
     It may be e.g. a message saying "Mr. User pressed Escape key
     and wishes the whole thing to be stopped". To stop the whole thing,
     Mr. User has to write a routine termed event handler which makes
     something if Escape (or other desired key) has been pressed. In Delphi
     environment, the event of pressing a key is called KeyDown event
     and the handler's name which is called automatically if this event
     occurs is FormKeyDown (see below). In our example the handler only
     sets flag Finito to true}
    Application.ProcessMessages;
  until Finito;
end;

procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  case Key of  {which key has been pressed}
    VK_ESCAPE: Finito:=true;
    .
    .
    VK_RIGHT : PutYourHandToYourHead;
    VK_SPACE : DoWhateverSimonSays;
    .
    . etc.
  end;
end;

end.






